% Copyright 2014 Sandia Corporation. Under the terms of Contract DE-AC04-94AL85000 with Sandia
% Corporation, the U.S. Government retains certain rights in this software
%
% This file is part of Sandia SPT (Sandia Simple Particle Tracking) v. 1.0.
% 
% Sandia SPT is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Sandia SPT is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Sandia SPT.  If not, see <http://www.gnu.org/licenses/>.

function displacements = calcDisplacement(sourcePos,sinkPos)

%function displacements = calcDisplacement(sourcePos,sinkPos)
%
%PURPOSE:
%	calcDisplacement is designed to calculate the displacement between
%	every pair of Cartesian coordinates in the first input and the second
%	input. 
%
%DEPENDENCIES:
%	-None-
%
%INPUTS:
%	sourcePos:
%		Each row corresponds to a signal detected in the source frame. 
%		Each column corresponds to one of the Cartesian coordinates. 
%	sinkPos:
%		Each row corresponds to a signal detected in the sink frame. 
%		Each column corresponds to one of the Cartesian coordinates. 
%
%OUTPUTS:
%	displacements:
%		The (i,j)th element of the output matrix will correspond to the
%		displacement from the (i)th position in sourcePos to the (j)th
%		position in sinkPos. 
%
%LIMITATIONS:
%	-None-
%
%NOTES:
%	sourcePos and sinkPos must have the same number of Cartesian coordinate
%	dimensions. 
%
%Written by Stephen M. Anthony 08/19/2014
%Last modified on 08/19/14 by Stephen M. Anthony

%Ensure that the dimensionality matches
if size(sourcePos,2) ~= size(sinkPos,2)
	error('calcDisplacement:DimensionMismatch',...
	'The two inputs must have the same number of dimensions')
else
	%Record the number of dimensions
	nDims = size(sourcePos,2);
end

%Determine the number of positions in each input
nSource = size(sourcePos,1);
nSink = size(sinkPos,1);

%Initialize a variable for the square of the displacement
dispSqr = zeros(nSource,nSink);

%Loop through all the dimensions
for dimIndex = 1:nDims
	%Extract the vector of position coordinates for this dimension from 
	%positions1, and replicate it nPos2 times
	thisP1 = sourcePos(:,dimIndex)*ones(1,nSink);
	%Extract the vector of position coordinates for this dimension from 
	%positions2, transpose it to convert it to a row, and replicate it 
	%nPos1 times. 
	thisP2 = ones(nSource,1)*sinkPos(:,dimIndex)';
	
	%Compute the displacement in this dimension
	thisDisp = thisP1 - thisP2;
	
	%Square the displacement and add it to the squared displacement
	%variable. 
	dispSqr = dispSqr + thisDisp.^2;
end

%Take the square root to obtain the actual displacement over all dimensions
displacements = sqrt(dispSqr);